<?php

namespace garmayev\calendar;

use garmayev\calendar\assets\CalendarAsset;
use yii\base\InvalidConfigException;
use yii\base\Widget;
use yii\helpers\Html;
use yii\helpers\Json;

/**
 * ReactCalendar Widget
 *
 * @property array $options HTML options for container
 * @property array $clientOptions React component options
 * @property string $inputName Name for hidden input (optional)
 * @property string $value Initial value
 */
class CalendarWidget extends Widget
{
    public $options = [];
    public $clientOptions = [];
    public $inputName;
    public $value;
    public $enableFormInput = false;

    /**
     * @var string Asset bundle class
     */
    public $assetBundle = CalendarAsset::class;

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();

        if (!isset($this->options['id'])) {
            $this->options['id'] = $this->getId();
        }

        $this->checkDependencies();
        $this->registerAssets();
    }

    /**
     * Проверяет наличие необходимых зависимостей
     */
    protected function checkDependencies()
    {
        $npmPath = \Yii::getAlias('@npm');

        if (!file_exists($npmPath)) {
            throw new InvalidConfigException(
                "NPM assets path (@npm) not found. Please make sure you have installed " .
                "the required npm packages: react, react-dom, react-calendar"
            );
        }

        $requiredPackages = [
            'react',
            'react-dom',
            'react-calendar'
        ];

        foreach ($requiredPackages as $package) {
            $packagePath = $npmPath . '/' . $package;
            if (!file_exists($packagePath)) {
                \Yii::warning(
                    "NPM package '{$package}' not found at {$packagePath}. " .
                    "Please install it using: npm install {$package}",
                    __METHOD__
                );
            }
        }
    }

    /**
     * Регистрирует ассеты
     */
    protected function registerAssets()
    {
        $assetBundle = $this->assetBundle;
        $assetBundle::register($this->getView());
    }

    /**
     * @inheritdoc
     */
    public function run()
    {
        $content = '';

        if ($this->enableFormInput && $this->inputName) {
            $content .= Html::hiddenInput($this->inputName, $this->value, [
                'id' => $this->options['id'] . '-input'
            ]);
        }

        $content .= Html::tag('div', '', $this->options);

        echo $content;
        $this->registerClientScript();
    }

    protected function registerClientScript()
    {
        $id = $this->options['id'];
        $inputId = $this->options['id'] . '-input';

        // Подготавливаем значение
        $value = null;
        if ($this->value) {
            if ($this->value instanceof \DateTime) {
                $value = $this->value->format('Y-m-d');
            } else {
                $value = (string) $this->value;
            }
        }

        $defaultOptions = [
            'value' => $value ? new \yii\web\JsExpression("new Date('{$value}')") : null,
        ];

        $options = array_merge($defaultOptions, $this->clientOptions);

        // Добавляем обработчик onChange если включен ввод формы
        if ($this->enableFormInput && $this->inputName) {
            $existingOnChange = $options['onChange'] ?? null;

            $options['onChange'] = new \yii\web\JsExpression("
                function(value) {
                    const input = document.getElementById('{$inputId}');
                    if (input && value) {
                        input.value = value.toISOString().split('T')[0];
                    }
                    " . ($existingOnChange ? "({$existingOnChange})(value);" : "") . "
                }
            ");
        }

        $jsonOptions = empty($options) ? '{}' : Json::encode($options);

        $js = <<<JS
            (function() {
                // Ждем загрузки DOM и React
                function initCalendar() {
                    const container = document.getElementById('{$id}');
                    if (container && window.React && window.ReactDOM && window.Calendar) {
                        window.ReactDOM.render(
                            React.createElement( Calendar, {$jsonOptions} ),
                            container
                        )
                    } else {
                        // Повторяем попытку через 100ms
                        setTimeout(initCalendar, 100);
                    }
                }
                
                if (document.readyState === 'loading') {
                    document.addEventListener('DOMContentLoaded', initCalendar);
                } else {
                    initCalendar();
                }
            })();
JS;

        $this->getView()->registerJs($js);
    }
}