<h1 align="center">
Yii2 bot telegram
</h1>
<p align="center">

<h2>Installation</h2>
------------

The preferred way to install this extension is through [composer](http://getcomposer.org/download/).

Either run

```
php composer.phar require garmayev/yii2-bot-telegram "*"
```

or add

```
"garmayev/yii2-bot-telegram": "*"
```

to the require section of your `composer.json` file.

Method list usable
-----
list methods

```
getMe
sendMessage
forwardMessage
sendPhoto
sendAudio
sendDocument
sendSticker
sendVideo
sendLocation
sendChatAction
getUserProfilePhotos
getUpdates
setWebhook
getChat
getChatAdministrators
getChatMembersCount
getChatMember
answerCallbackQuery
editMessageText
editMessageCaption
sendGame
Game
Animation
CallbackGame
getGameHighScores
GameHighScore
answerInlineQuery
setChatStickerSet
deleteChatStickerSet
leaveChat
pinChatMessage
unpinChatMessage
setChatDescription
setChatTitle
deleteChatPhoto 
exportChatInviteLink 
promoteChatMember
restrictChatMember
unbanChatMember
kickChatMember
editMessageLiveLocation
stopMessageLiveLocation
```

Usage
-----
first add to config.php

```php
<?php
'components' => [
	'telegram' => [
		'class' => 'garmayev\telegram\Telegram',
		'botToken' => '112488045:AAGs6CVXgaqC92pvt1u0L6Azfsdfd',
	]
]
?>
```

Once the extension is installed, simply use it in your code by  :

```php
<?php Yii::$app->telegram->sendMessage([
	'chat_id' => $chat_id,
	'text' => 'test',
]); ?>
```

send message width inline keyboard by:

```php
<?php Yii::$app->telegram->sendMessage([
        'chat_id' => $chat_id,
        'text' => 'this is test',
        'reply_markup' => json_encode([
            'inline_keyboard'=>[
                [
                    ['text'=>"refresh",'callback_data'=> time()]
                ]
            ]
        ]),
    ]); ?>
```

send photo by :

```php
<?php 
Yii::$app->telegram->sendPhoto([
	'chat_id' => $chat_id,
	'photo' => Yii::$app->getBaseUrl().'/uploads/test.jpg',
	'caption' => 'this is test'
]); ?>
```

<h2>Usage in controller</h2>
-----

First of all you need to disable the enableCsrfValidation feature in the controller

The robot is currently running from your server
But when we start /start run the robot from the telegram application on the mobile, the request does not reach the
action inside the controller because the telegram sends the request to the POST and yii requests it without csrf
Sends Bad Request (# 400). So then the code doesn't run inside your method

Consider the following example

```php
class SiteController extends Controller
{
	public $enableCsrfValidation = false;

	public function actionIndex()
    {
        $res = Yii::$app->telegram->sendMessage([
            'chat_id' => $chat_id,
            'text' => 'hello world!!' 
        ]);
       
    }
}
```

### How to get user chat_id from the bot ?

> __You can use : `$telegram->input->message->chat->id` to get chat_id__

## Примеры реализации обработчиков контента:

### ContactHandler.php

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use Yii;

class ContactHandler implements ContentHandlerInterface
{
    public static function getContentType()
    {
        return \garmayev\telegram\base\Command::TYPE_CONTACT;
    }

    public function handle($telegram, $contactData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;
        $contact = $contactData;

        // Сохраняем контакт в базе данных
        $phone = $contact->phone_number;
        $firstName = $contact->first_name;
        $lastName = $contact->last_name ?? '';
        $userId = $contact->user_id ?? null;

        Yii::info("Received contact: {$firstName} {$lastName} - {$phone}", 'telegram');

        // Отправляем подтверждение
        return $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => "✅ Контакт получен!\n📱 Телефон: {$phone}\n👤 Имя: {$firstName}",
            'parse_mode' => 'HTML'
        ]);
    }
}
```

### LocationHandler.php

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use Yii;

class LocationHandler implements ContentHandlerInterface
{
    public static function getContentType()
    {
        return \garmayev\telegram\base\Command::TYPE_LOCATION;
    }

    public function handle($telegram, $locationData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;
        $location = $locationData;

        $latitude = $location->latitude;
        $longitude = $location->longitude;

        Yii::info("Received location: {$latitude}, {$longitude}", 'telegram');

        // Сохраняем локацию в базе данных
        // $this->saveUserLocation($chatId, $latitude, $longitude);

        // Отправляем подтверждение и карту
        $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => "📍 Локация получена!\nШирота: {$latitude}\nДолгота: {$longitude}"
        ]);

        // Отправляем карту с меткой
        return $telegram->sendLocation([
            'chat_id' => $chatId,
            'latitude' => $latitude,
            'longitude' => $longitude
        ]);
    }
}
```

### PhotoHandler.php

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use Yii;

class PhotoHandler implements ContentHandlerInterface
{
    public static function getContentType()
    {
        return \garmayev\telegram\base\Command::TYPE_PHOTO;
    }

    public function handle($telegram, $photoData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;
        $messageId = $telegram->input->message->message_id;
        $caption = $telegram->input->message->caption ?? '';

        // Фото приходит как массив с разными размерами, берем самое большое (последнее)
        $photo = end($photoData);
        $fileId = $photo->file_id;

        Yii::info("Received photo: {$fileId}, caption: {$caption}", 'telegram');

        // Скачиваем файл
        $file = $telegram->getFile(['file_id' => $fileId]);
        if ($file && isset($file->file_path)) {
            $fileUrl = "https://api.telegram.org/file/bot{$telegram->token}/{$file->file_path}";
            // $this->downloadAndProcessImage($fileUrl, $chatId, $caption);
        }

        // Отправляем подтверждение
        return $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => "📸 Фото получено!" . ($caption ? "\n📝 Подпись: {$caption}" : ""),
            'reply_to_message_id' => $messageId
        ]);
    }
}
```

### DocumentHandler.php

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use Yii;

class DocumentHandler implements ContentHandlerInterface
{
    public static function getContentType()
    {
        return \garmayev\telegram\base\Command::TYPE_DOCUMENT;
    }

    public function handle($telegram, $documentData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;
        $messageId = $telegram->input->message->message_id;
        
        $document = $documentData;
        $fileId = $document->file_id;
        $fileName = $document->file_name ?? 'unknown';
        $mimeType = $document->mime_type ?? 'unknown';
        $fileSize = $document->file_size ?? 0;

        Yii::info("Received document: {$fileName} ({$mimeType}, {$fileSize} bytes)", 'telegram');

        // Проверяем тип файла
        $allowedTypes = ['application/pdf', 'image/jpeg', 'image/png', 'text/plain'];
        if (!in_array($mimeType, $allowedTypes)) {
            return $telegram->sendMessage([
                'chat_id' => $chatId,
                'text' => "❌ Формат файла не поддерживается. Разрешены: PDF, JPEG, PNG, TXT",
                'reply_to_message_id' => $messageId
            ]);
        }

        // Проверяем размер файла (5MB)
        if ($fileSize > 5 * 1024 * 1024) {
            return $telegram->sendMessage([
                'chat_id' => $chatId,
                'text' => "❌ Файл слишком большой. Максимальный размер: 5MB",
                'reply_to_message_id' => $messageId
            ]);
        }

        // Скачиваем файл
        $file = $telegram->getFile(['file_id' => $fileId]);
        if ($file && isset($file->file_path)) {
            $fileUrl = "https://api.telegram.org/file/bot{$telegram->token}/{$file->file_path}";
            // $this->processDocument($fileUrl, $fileName, $mimeType, $chatId);
        }

        return $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => "📄 Документ получен!\n📝 Имя: {$fileName}\n📊 Размер: " . round($fileSize / 1024, 2) . " KB",
            'reply_to_message_id' => $messageId
        ]);
    }
}
```

### VideoHandler.php

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use Yii;

class VideoHandler implements ContentHandlerInterface
{
    public static function getContentType()
    {
        return \garmayev\telegram\base\Command::TYPE_VIDEO;
    }

    public function handle($telegram, $videoData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;
        $messageId = $telegram->input->message->message_id;
        $caption = $telegram->input->message->caption ?? '';

        $video = $videoData;
        $fileId = $video->file_id;
        $duration = $video->duration ?? 0;
        $width = $video->width ?? 0;
        $height = $video->height ?? 0;
        $fileSize = $video->file_size ?? 0;

        Yii::info("Received video: {$duration}s, {$width}x{$height}, {$fileSize} bytes", 'telegram');

        return $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => "🎥 Видео получено!\n⏱ Длительность: {$duration} сек\n📏 Размер: {$width}x{$height}" . 
                     ($caption ? "\n📝 Подпись: {$caption}" : ""),
            'reply_to_message_id' => $messageId
        ]);
    }
}
```

### Конфигурация в модуле:

```php
<?php
return [
    'components' => [
        'telegram' => [
            'class' => 'garmayev\telegram\Telegram',
            'token' => 'YOUR_BOT_TOKEN',
            'commandConfig' => [
                'commands' => [
                    '/start' => 'app\telegram\commands\StartCommand',
                    '/help' => 'app\telegram\commands\HelpCommand',
                ],
                'contentHandlers' => [
                    \garmayev\telegram\base\Command::TYPE_CONTACT => 'app\telegram\handlers\ContactHandler',
                    \garmayev\telegram\base\Command::TYPE_LOCATION => 'app\telegram\handlers\LocationHandler',
                    \garmayev\telegram\base\Command::TYPE_PHOTO => 'app\telegram\handlers\PhotoHandler',
                    \garmayev\telegram\base\Command::TYPE_DOCUMENT => 'app\telegram\handlers\DocumentHandler',
                    \garmayev\telegram\base\Command::TYPE_VIDEO => 'app\telegram\handlers\VideoHandler',
                    \garmayev\telegram\base\Command::TYPE_VOICE => 'app\telegram\handlers\VoiceHandler',
                ],
                'scanNamespaces' => [
                    'app\telegram\commands',
                    'app\telegram\handlers',
                ],
            ],
        ],
    ],
];
```

### Использование в контроллере:

```php
<?php
namespace app\controllers;

use yii\web\Controller;
use garmayev\telegram\base\Command;

class TelegramController extends Controller
{
    public function actionWebhook()
    {
        // Обработка всех входящих сообщений
        $result = Command::handle();
        
        if ($result === false) {
            // Логирование ошибок
            Yii::error('Error processing Telegram update', 'telegram');
        }
        
        return 'ok';
    }
}
```

### Основные улучшения:

- Универсальная система обработки - поддержка всех типов контента Telegram
- Интерфейс для обработчиков - единый интерфейс ContentHandlerInterface
- Автоматическое определение типа - метод detectContentType() определяет тип входящего контента
- Гибкая конфигурация - регистрация обработчиков через конфиг
- Обработка медиафайлов - фото, видео, документы, голосовые сообщения
- Информативное логирование - детальное логирование полученного контента
- Обработка ошибок - проверка типов файлов, размеров и т.д.
- Масштабируемость - легко добавлять новые обработчики для других типов контента

Теперь бот может обрабатывать не только текстовые команды, но и любые типы контента, которые отправляют пользователи.

