<?php
namespace garmayev\telegram\base;

use Closure;
use Yii;
use yii\base\Component;
use yii\base\InvalidConfigException;
use yii\di\Container;
use yii\di\NotInstantiableException;

/**
 * @author Bato Garmayev <garmayev.ba@gmail.com>
 */
class Command extends Component
{
    use CallbackDataTrait;

    /**
     * @var array зарегистрированные команды
     */
    private static $registeredCommands = [];

    /**
     * @var array зарегистрированные обработчики контента
     */
    private static $contentHandlers = [];

    /**
     * @var array зарегистрированные обработчики callback-запросов
     */
    private static $callbackHandlers = [];

    /**
     * @var Container DI контейнер для создания объектов
     */
    private static $container;

    /**
     * @var array команды для установки в боте
     */
    private static $botCommands = [];

    /**
     * @var bool флаг инициализации
     */
    private static $initialized = false;

    // Типы контента
    const TYPE_TEXT = 'text';
    const TYPE_CONTACT = 'contact';
    const TYPE_LOCATION = 'location';
    const TYPE_DOCUMENT = 'document';
    const TYPE_PHOTO = 'photo';
    const TYPE_VIDEO = 'video';
    const TYPE_AUDIO = 'audio';
    const TYPE_VOICE = 'voice';
    const TYPE_VIDEO_NOTE = 'video_note';
    const TYPE_STICKER = 'sticker';
    const TYPE_ANIMATION = 'animation';
    const TYPE_POLL = 'poll';
    const TYPE_DICE = 'dice';
    const TYPE_CALLBACK_QUERY = 'callback_query';

    /**
     * Инициализация команд из конфигурации компонента telegram
     */
    public static function initialize()
    {
        if (self::$initialized) {
            return;
        }

        if (!Yii::$app->has('telegram')) {
            Yii::warning('Telegram component not found in application configuration', 'telegram');
            return;
        }

        $telegram = Yii::$app->telegram;

        // Инициализация команд из конфигурации компонента
        if (isset($telegram->commands) && is_array($telegram->commands)) {
            foreach ($telegram->commands as $command => $handler) {
                self::register($command, $handler);
            }
        }

        // Регистрация обработчиков контента
        if (isset($telegram->contentHandlers) && is_array($telegram->contentHandlers)) {
            foreach ($telegram->contentHandlers as $type => $handler) {
                self::registerContentHandler($type, $handler);
            }
        }

        // Регистрация обработчиков callback-запросов
        if (isset($telegram->callbackHandlers) && is_array($telegram->callbackHandlers)) {
            foreach ($telegram->callbackHandlers as $callbackData => $handler) {
                self::registerCallbackHandler($callbackData, $handler);
            }
        }

        // Автоматическая регистрация через сканирование namespace
        if (isset($telegram->scanNamespaces) && is_array($telegram->scanNamespaces)) {
            foreach ($telegram->scanNamespaces as $namespace) {
                self::scanNamespace($namespace);
            }
        }

        // Сохранение команд для бота
        if (isset($telegram->botCommands) && is_array($telegram->botCommands)) {
            self::$botCommands = $telegram->botCommands;
        }

        // Настройка DI контейнера
        if (isset($telegram->container) && $telegram->container instanceof Container) {
            self::$container = $telegram->container;
        }

        self::$initialized = true;
    }

    /**
     * Сканирование namespace для автоматической регистрации команд
     * @param string $namespace
     * @throws InvalidConfigException
     */
    private static function scanNamespace($namespace)
    {
        $alias = '@' . str_replace('\\', '/', $namespace);
        $path = Yii::getAlias($alias);

        if (!is_dir($path)) {
            Yii::warning("Namespace path does not exist: {$path}", 'telegram');
            return;
        }

        $files = scandir($path);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..' || !str_ends_with($file, '.php')) {
                continue;
            }

            $className = $namespace . '\\' . substr($file, 0, -4);

            // Регистрация команд
            if (class_exists($className) && is_subclass_of($className, ContentHandlerInterface::class)) {
                try {
                    $commandName = $className::getCommand();
                    if ($commandName) {
                        self::register($commandName, $className);
                    }
                } catch (\Exception $e) {
                    Yii::error("Failed to register command {$className}: " . $e->getMessage(), 'telegram');
                }
            }

            // Регистрация обработчиков контента
            if (class_exists($className) && is_subclass_of($className, ContentHandlerInterface::class)) {
                try {
                    $contentType = $className::getContentType();
                    self::registerContentHandler($contentType, $className);
                } catch (\Exception $e) {
                    Yii::error("Failed to register content handler {$className}: " . $e->getMessage(), 'telegram');
                }
            }

            // Регистрация обработчиков callback-запросов
            if (class_exists($className) && is_subclass_of($className, CallbackHandlerInterface::class)) {
                try {
                    $callbackData = $className::getCallbackData();
                    if ($callbackData) {
                        self::registerCallbackHandler($callbackData, $className);
                    }
                } catch (\Exception $e) {
                    Yii::error("Failed to register callback handler {$className}: " . $e->getMessage(), 'telegram');
                }
            }
        }
    }

    /**
     * Регистрация класса-обработчика для команды
     * @param string $command команда
     * @param string|array $handler класс обработчика или конфиг
     * @throws InvalidConfigException
     */
    public static function register($command, $handler)
    {
        if (is_string($handler) && !class_exists($handler)) {
            throw new InvalidConfigException("Command handler class {$handler} does not exist");
        }

        if (is_array($handler) && !isset($handler['class'])) {
            throw new InvalidConfigException("Command handler configuration must contain 'class' key");
        }

        // Проверяем, что обработчик реализует ContentHandlerInterface
        $className = is_string($handler) ? $handler : $handler['class'];
        if (!is_subclass_of($className, ContentHandlerInterface::class)) {
            throw new InvalidConfigException("Command handler {$className} must implement ContentHandlerInterface");
        }

        self::$registeredCommands[$command] = $handler;
    }

    /**
     * Регистрация обработчика контента
     * @param string $type тип контента
     * @param string|array $handler класс обработчика или конфиг
     * @throws InvalidConfigException
     */
    public static function registerContentHandler($type, $handler)
    {
        if (is_string($handler) && !class_exists($handler)) {
            throw new InvalidConfigException("Content handler class {$handler} does not exist");
        }

        if (is_array($handler) && !isset($handler['class'])) {
            throw new InvalidConfigException("Content handler configuration must contain 'class' key");
        }

        // Проверяем, что обработчик реализует ContentHandlerInterface
        $className = is_string($handler) ? $handler : $handler['class'];
        if (!is_subclass_of($className, ContentHandlerInterface::class)) {
            throw new InvalidConfigException("Content handler {$className} must implement ContentHandlerInterface");
        }

        self::$contentHandlers[$type] = $handler;
    }

    /**
     * Регистрация обработчика callback-запроса
     * @param string $callbackData callback data (может быть регулярным выражением)
     * @param string|array $handler класс обработчика или конфиг
     * @throws InvalidConfigException
     */
    public static function registerCallbackHandler($callbackData, $handler)
    {
        if (is_string($handler) && !class_exists($handler)) {
            throw new InvalidConfigException("Callback handler class {$handler} does not exist");
        }

        if (is_array($handler) && !isset($handler['class'])) {
            throw new InvalidConfigException("Callback handler configuration must contain 'class' key");
        }

        // Проверяем, что обработчик реализует CallbackHandlerInterface
        $className = is_string($handler) ? $handler : $handler['class'];
        if (!is_subclass_of($className, CallbackHandlerInterface::class)) {
            throw new InvalidConfigException("Callback handler {$className} must implement CallbackHandlerInterface");
        }

        self::$callbackHandlers[$callbackData] = $handler;
    }

    /**
     * Обработка callback-запроса
     * @param object $callbackQuery
     * @return mixed
     */
    private static function handleCallbackQuery($callbackQuery)
    {
        if (!isset($callbackQuery->data)) {
            return false;
        }

        $callbackData = $callbackQuery->data;
        $telegram = Yii::$app->telegram;

        // Поиск подходящего обработчика
        foreach (self::$callbackHandlers as $pattern => $handlerConfig) {
            // Если pattern - регулярное выражение
            if (str_starts_with($pattern, '/') && str_ends_with($pattern, '/')) {
                if (preg_match($pattern, $callbackData, $matches)) {
                    $parsedData = self::parseCallbackData($callbackData);
                    $handler = self::createHandler($handlerConfig);
                    return $handler->handleCallback($telegram, $callbackQuery, $parsedData['params']);
                }
            }
            // Если точное совпадение
            elseif ($pattern === $callbackData) {
                $parsedData = self::parseCallbackData($callbackData);
                $handler = self::createHandler($handlerConfig);
                return $handler->handleCallback($telegram, $callbackQuery, $parsedData['params']);
            }
            // Если pattern - префикс
            elseif (strpos($callbackData, $pattern . ':') === 0) {
                $parsedData = self::parseCallbackData($callbackData);
                $handler = self::createHandler($handlerConfig);
                return $handler->handleCallback($telegram, $callbackQuery, $parsedData['params']);
            }
        }

        Yii::warning("No callback handler found for: {$callbackData}", 'telegram');
        return false;
    }

    /**
     * Запуск обработки всех зарегистрированных команд и контента
     * @return mixed|void
     */
    public static function handle()
    {
        self::initialize();

        $telegram = Yii::$app->telegram;
        $input = $telegram->input;

        // Обработка callback-запросов в первую очередь
        if (isset($input->callback_query)) {
            return self::handleCallbackQuery($input->callback_query);
        }

        // Проверяем, есть ли следующий обработчик из предыдущей сессии
        $nextHandler = self::getNextHandler($input);
        if ($nextHandler) {
            try {
                self::clearNextHandler($input);
                $handler = self::createHandler($nextHandler['class']);

                // Если обработчик поддерживает состояние, загружаем его
                if ($handler instanceof StatefulHandlerInterface) {
                    $state = self::loadHandlerState(get_class($handler), $input);
                    $handler->setState($state);
                }

                $result = $handler->handleContent($telegram, $nextHandler['params'], $input);

                // Сохраняем состояние обработчика, если нужно
                if ($handler instanceof StatefulHandlerInterface) {
                    self::saveHandlerState(get_class($handler), $handler->getState(), $input);
                }

                return $result;
            } catch (\Exception $e) {
                Yii::error("Error executing next handler {$nextHandler['class']}: " . $e->getMessage(), 'telegram');
                // Продолжаем обычную обработку в случае ошибки
            }
        }

        // Определяем тип контента
        $contentType = self::detectContentType($input);
        if ($contentType === null) {
            return;
        }

        // Получаем данные контента
        $contentData = self::getContentData($input, $contentType);
        if ($contentData === null) {
            return;
        }

        $isCallback = isset($input->callback_query);
        $callbackQuery = $isCallback ? $input->callback_query : null;

        // Обработка текстовых команд
        if ($contentType === self::TYPE_TEXT) {
            $args = explode(' ', $contentData);
            $inputCommand = array_shift($args);

            // Проверяем зарегистрированные команды
            if (isset(self::$registeredCommands[$inputCommand])) {
                try {
                    $handler = self::createHandler(self::$registeredCommands[$inputCommand]);

                    // Если обработчик поддерживает состояние, загружаем его
                    if ($handler instanceof StatefulHandlerInterface) {
                        $state = self::loadHandlerState(get_class($handler), $input);
                        $handler->setState($state);
                    }

                    $result = $handler->execute($telegram, $args, $isCallback, $callbackQuery);

                    // Сохраняем состояние обработчика, если нужно
                    if ($handler instanceof StatefulHandlerInterface) {
                        self::saveHandlerState(get_class($handler), $handler->getState(), $input);
                    }

                    return $result;
                } catch (\Exception $e) {
                    Yii::error("Error executing command {$inputCommand}: " . $e->getMessage(), 'telegram');
                    return false;
                }
            }
        }

        // Обработка контента
        if (isset(self::$contentHandlers[$contentType])) {
            try {
                $handler = self::createHandler(self::$contentHandlers[$contentType]);

                // Если обработчик поддерживает состояние, загружаем его
                if ($handler instanceof StatefulHandlerInterface) {
                    $state = self::loadHandlerState(get_class($handler), $input);
                    $handler->setState($state);
                }

                $result = $handler->handle($telegram, $contentData, $isCallback, $callbackQuery);

                // Сохраняем состояние обработчика, если нужно
                if ($handler instanceof StatefulHandlerInterface) {
                    self::saveHandlerState(get_class($handler), $handler->getState(), $input);
                }

                return $result;
            } catch (\Exception $e) {
                Yii::error("Error handling content type {$contentType}: " . $e->getMessage(), 'telegram');
                return false;
            }
        }

        // Если нет обработчика для данного типа контента
        Yii::warning("No handler registered for content type: {$contentType}", 'telegram');
        return false;
    }

    // ... остальные методы остаются без изменений

    /**
     * Получить список зарегистрированных обработчиков callback-запросов
     * @return array
     */
    public static function getCallbackHandlers()
    {
        self::initialize();
        return array_keys(self::$callbackHandlers);
    }

    /**
     * Проверить, зарегистрирован ли обработчик для callback data
     * @param string $callbackData
     * @return bool
     */
    public static function hasCallbackHandler($callbackData)
    {
        self::initialize();
        return isset(self::$callbackHandlers[$callbackData]);
    }

    /**
     * Очистить все обработчики callback-запросов
     */
    public static function clearCallbackHandlers()
    {
        self::$callbackHandlers = [];
    }
}