## Пример использования - создание обработчика для инлайновых кнопок:

```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\CallbackHandlerInterface;
use garmayev\telegram\base\CallbackDataTrait;
use Yii;

/**
 * Пример обработчика для инлайновых кнопок
 */
class ExampleCallbackHandler implements CallbackHandlerInterface
{
    use CallbackDataTrait;

    /**
     * @inheritDoc
     */
    public static function getCallbackData()
    {
        return 'example'; // Обрабатывает callback data начинающиеся с 'example:'
    }

    /**
     * @inheritDoc
     */
    public function handleCallback($telegram, $callbackQuery, $params = [])
    {
        $chatId = $callbackQuery->message->chat->id;
        $messageId = $callbackQuery->message->message_id;
        $callbackQueryId = $callbackQuery->id;

        $action = $params[0] ?? 'default';

        switch ($action) {
            case 'button1':
                // Обработка нажатия на первую кнопку
                $telegram->editMessageText([
                    'chat_id' => $chatId,
                    'message_id' => $messageId,
                    'text' => 'Вы нажали на кнопку 1!'
                ]);
                break;

            case 'button2':
                // Обработка нажатия на вторую кнопку
                $telegram->editMessageText([
                    'chat_id' => $chatId,
                    'message_id' => $messageId,
                    'text' => 'Вы нажали на кнопку 2!'
                ]);
                break;

            default:
                // Ответ по умолчанию
                $telegram->answerCallbackQuery([
                    'callback_query_id' => $callbackQueryId,
                    'text' => 'Действие выполнено!'
                ]);
        }

        return true;
    }
}
```

## Пример создания клавиатуры с инлайновыми кнопками:

```php
<?php
namespace app\telegram\commands;

use garmayev\telegram\base\ContentHandlerInterface;
use garmayev\telegram\base\CallbackDataTrait;
use Yii;

class ExampleCommand implements ContentHandlerInterface
{
    use CallbackDataTrait;

    /**
     * @inheritDoc
     */
    public static function getCommand()
    {
        return '/example';
    }

    /**
     * @inheritDoc
     */
    public static function getContentType()
    {
        return 'text';
    }

    /**
     * @inheritDoc
     */
    public function handle($telegram, $contentData, $isCallback, $callbackQuery = null)
    {
        $chatId = $telegram->input->message->chat->id;

        // Создаем инлайновую клавиатуру
        $keyboard = [
            'inline_keyboard' => [
                [
                    [
                        'text' => 'Кнопка 1',
                        'callback_data' => self::buildCallbackData('example', ['button1'])
                    ],
                    [
                        'text' => 'Кнопка 2', 
                        'callback_data' => self::buildCallbackData('example', ['button2'])
                    ]
                ]
            ]
        ];

        $telegram->sendMessage([
            'chat_id' => $chatId,
            'text' => 'Выберите действие:',
            'reply_markup' => json_encode($keyboard)
        ]);

        return true;
    }
}
```

## Конфигурация в config.php:

```php
'components' => [
    'telegram' => [
        'class' => 'garmayev\telegram\Telegram',
        'botToken' => 'YOUR_BOT_TOKEN',
        'commands' => [
            '/example' => 'app\telegram\commands\ExampleCommand',
        ],
        'callbackHandlers' => [
            'example' => 'app\telegram\handlers\ExampleCallbackHandler',
        ],
        'scanNamespaces' => [
            'app\telegram\commands',
            'app\telegram\handlers',
        ],
    ],
],
```

### Эти изменения позволяют:

- Создавать отдельные классы-обработчики для инлайновых кнопок
- Автоматически регистрировать обработчики через конфигурацию или сканирование namespace
- Обрабатывать callback data с параметрами
- Поддерживать различные паттерны callback data (точное совпадение, префикс, регулярные выражения)
- Интегрировать обработку callback-запросов в существующую архитектуру бота

Обработчики callback-запросов имеют приоритет над другими типами контента и выполняются в первую очередь при получении callback_query от Telegram API.

