# Документация по классу Command для Telegram бота
## Обзор
Класс Command предоставляет мощную систему для обработки команд и контента в Telegram боте на Yii2. Он поддерживает регистрацию обработчиков, управление контекстом через сессии, цепочки обработчиков и многое другое.

## Быстрый старт
### Базовая настройка

```php
use garmayev\telegram\base\Command;

// Инициализация в конфигурации приложения
Command::initCommands([
    'commands' => [
        '/start' => app\telegram\commands\StartHandler::class,
        '/help' => app\telegram\commands\HelpHandler::class,
    ],
    'contentHandlers' => [
        Command::TYPE_TEXT => app\telegram\handlers\TextHandler::class,
        Command::TYPE_PHOTO => app\telegram\handlers\PhotoHandler::class,
    ],
    'scanNamespaces' => [
        'app\telegram\commands',
        'app\telegram\handlers',
    ],
]);
```

### Обработка входящих сообщений
```php
// В контроллере или точке входа
$result = Command::handle();
```
## Создание обработчиков
### Базовый обработчик команды
```php
namespace app\telegram\commands;

use garmayev\telegram\base\ContentHandlerInterface;
use yii\base\BaseObject;

class StartHandler extends BaseObject implements ContentHandlerInterface
{
public static function getCommand()
{
return '/start';
}

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => 'Добро пожаловать! Я ваш бот.',
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [
                        ['text' => 'Помощь', 'callback_data' => '/help'],
                        ['text' => 'О нас', 'callback_data' => '/about']
                    ]
                ]
            ])
        ]);
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        // Не используется для команд, только для обработчиков контента
        return null;
    }
}
```
### Обработчик с состоянием
```php
namespace app\telegram\commands;

use garmayev\telegram\base\ContentHandlerInterface;
use garmayev\telegram\base\StatefulHandlerInterface;
use yii\base\BaseObject;

class RegistrationHandler extends BaseObject implements ContentHandlerInterface, StatefulHandlerInterface
{
private $state = [];

    public static function getCommand()
    {
        return '/register';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function setState(array $state)
    {
        $this->state = $state;
    }

    public function getState()
    {
        return $this->state;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        $step = $this->state['step'] ?? 0;

        switch ($step) {
            case 0:
                $this->state['step'] = 1;
                Command::setContext('awaiting_name', true);
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Давайте зарегистрируем вас! Введите ваше имя:'
                ]);
            
            case 1:
                $name = $args[0] ?? '';
                if (empty($name)) {
                    return $telegram->sendMessage([
                        'chat_id' => $telegram->getChatId(),
                        'text' => 'Пожалуйста, введите корректное имя:'
                    ]);
                }

                $this->state['name'] = $name;
                $this->state['step'] = 2;
                Command::removeContext('awaiting_name');
                Command::setContext('awaiting_email', true);
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => "Отлично, {$name}! Теперь введите ваш email:"
                ]);
            
            case 2:
                $email = $args[0] ?? '';
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    return $telegram->sendMessage([
                        'chat_id' => $telegram->getChatId(),
                        'text' => 'Пожалуйста, введите корректный email:'
                    ]);
                }

                $this->state['email'] = $email;
                Command::removeContext('awaiting_email');
                
                // Завершаем регистрацию
                $message = "Регистрация завершена!\n\n";
                $message .= "Имя: {$this->state['name']}\n";
                $message .= "Email: {$this->state['email']}";
                
                // Очищаем состояние
                $this->state = [];
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => $message
                ]);
        }
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

### Обработчик контента (фото)
```php
namespace app\telegram\handlers;

use garmayev\telegram\base\ContentHandlerInterface;
use yii\base\BaseObject;

class PhotoHandler extends BaseObject implements ContentHandlerInterface
{
    public static function getCommand()
    {
        return null; // Не используется для обработчиков контента
    }

    public static function getContentType()
    {
        return Command::TYPE_PHOTO;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        return null; // Не используется для обработчиков контента
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        // $contentData содержит массив с информацией о фото
        $photo = $contentData[count($contentData) - 1]; // Берем фото наибольшего размера
        $fileId = $photo->file_id;
        
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => "Спасибо за фото! File ID: {$fileId}"
        ]);
    }
}
```

## Работа с контекстом
### Управление контекстом
```php
// Установка значения в контекст
Command::setContext('current_step', 'awaiting_phone');

// Получение значения из контекста
$step = Command::getContext('current_step', 'default_value');

// Удаление значения из контекста
Command::removeContext('current_step');

// Получение всего контекста
$context = Command::getAllContext();

// Очистка всего контекста
Command::clearContext();
```

### Пример использования контекста
```php
class OrderHandler implements ContentHandlerInterface
{
    public static function getCommand()
    {
        return '/order';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        // Проверяем, есть ли активный заказ в контексте
        $activeOrder = Command::getContext('active_order');
        
        if ($activeOrder) {
            return $this->continueOrder($telegram, $args);
        } else {
            return $this->startNewOrder($telegram, $args);
        }
    }

    private function startNewOrder($telegram, $args)
    {
        Command::setContext('active_order', true);
        Command::setContext('order_step', 'product_selection');
        
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => 'Какой товар вас интересует?',
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [
                        ['text' => 'Товар 1', 'callback_data' => 'product_1'],
                        ['text' => 'Товар 2', 'callback_data' => 'product_2']
                    ]
                ]
            ])
        ]);
    }

    private function continueOrder($telegram, $args)
    {
        $step = Command::getContext('order_step');
        
        switch ($step) {
            case 'product_selection':
                // Обработка выбора товара
                $product = $args[0] ?? '';
                Command::setContext('selected_product', $product);
                Command::setContext('order_step', 'quantity');
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Введите количество:'
                ]);
            
            case 'quantity':
                // Обработка ввода количества
                $quantity = $args[0] ?? '';
                Command::setContext('quantity', $quantity);
                
                // Завершение заказа
                $product = Command::getContext('selected_product');
                
                // Очищаем контекст заказа
                Command::removeContext('active_order');
                Command::removeContext('order_step');
                Command::removeContext('selected_product');
                Command::removeContext('quantity');
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => "Заказ оформлен!\nТовар: {$product}\nКоличество: {$quantity}"
                ]);
        }
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

## Цепочки обработчиков
### Создание цепочки обработчиков
```php
class SurveyHandler implements ContentHandlerInterface, StatefulHandlerInterface
{
    private $state = [];

    public static function getCommand()
    {
        return '/survey';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function setState(array $state)
    {
        $this->state = $state;
    }

    public function getState()
    {
        return $this->state;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        $step = $this->state['step'] ?? 0;

        switch ($step) {
            case 0:
                $this->state['step'] = 1;
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Вопрос 1: Как вас зовут?'
                ]);
            
            case 1:
                $this->state['name'] = $args[0] ?? '';
                $this->state['step'] = 2;
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Вопрос 2: Сколько вам лет?'
                ]);
            
            case 2:
                $this->state['age'] = $args[0] ?? '';
                
                // Передаем управление следующему обработчику
                Command::setNextHandler(ResultsHandler::class, [
                    'name' => $this->state['name'],
                    'age' => $this->state['age']
                ]);
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Спасибо за ответы! Обрабатываем результаты...'
                ]);
        }
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}

class ResultsHandler implements ContentHandlerInterface
{
public static function getCommand()
{
return null;
}

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        return null;
    }

    public function handleContent($telegram, $params, $input)
    {
        $name = $params['name'] ?? 'Не указано';
        $age = $params['age'] ?? 'Не указано';
        
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => "Результаты опроса:\nИмя: {$name}\nВозраст: {$age}"
        ]);
    }
}
```

### Обработка callback запросов
```php
class MenuHandler implements ContentHandlerInterface
{
public static function getCommand()
{
return '/menu';
}

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        if ($isCallback) {
            return $this->handleCallback($telegram, $callbackQuery);
        }

        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => 'Выберите действие:',
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [
                        ['text' => 'Опция 1', 'callback_data' => 'option_1'],
                        ['text' => 'Опция 2', 'callback_data' => 'option_2']
                    ],
                    [
                        ['text' => 'Информация', 'callback_data' => 'info']
                    ]
                ]
            ])
        ]);
    }

    private function handleCallback($telegram, $callbackQuery)
    {
        $data = $callbackQuery->data;
        $messageId = $callbackQuery->message->message_id;
        $chatId = $callbackQuery->message->chat->id;

        switch ($data) {
            case 'option_1':
                $text = 'Вы выбрали опцию 1';
                break;
            case 'option_2':
                $text = 'Вы выбрали опцию 2';
                break;
            case 'info':
                $text = 'Это информационное сообщение';
                break;
            default:
                $text = 'Неизвестная опция';
        }

        // Ответ на callback запрос
        $telegram->answerCallbackQuery([
            'callback_query_id' => $callbackQuery->id,
            'text' => 'Обработано'
        ]);

        // Обновление сообщения
        return $telegram->editMessageText([
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text
        ]);
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

### Автоматическая регистрация через сканирование namespace
```php
// В конфигурации
Command::initCommands([
    'scanNamespaces' => [
    'app\telegram\commands',
    'app\telegram\handlers',
],
]);

// Структура папок:
// - app/
//   - telegram/
//     - commands/
//       - StartHandler.php
//       - HelpHandler.php
//     - handlers/
//       - TextHandler.php
//       - PhotoHandler.php
```

## Утилиты и вспомогательные методы
### Получение информации о зарегистрированных командах
```php
// Все зарегистрированные команды
$commands = Command::getRegisteredCommands();

// Все обработчики контента
$handlers = Command::getContentHandlers();

// Проверка регистрации команды
if (Command::isRegistered('/start')) {
    // Команда зарегистрирована
}

// Проверка наличия обработчика для типа контента
if (Command::hasContentHandler(Command::TYPE_PHOTO)) {
    // Обработчик зарегистрирован
}
```

### Очистка регистраций
```php
// Очистка всех команд
Command::clearCommands();

// Очистка всех обработчиков контента
Command::clearContentHandlers();
```

## Типы контента
### Доступные константы для типов контента:

```
Command::TYPE_TEXT - текстовые сообщения
Command::TYPE_PHOTO - фото
Command::TYPE_VIDEO - видео
Command::TYPE_DOCUMENT - документы
Command::TYPE_AUDIO - аудио
Command::TYPE_VOICE - голосовые сообщения
Command::TYPE_LOCATION - локации
Command::TYPE_CONTACT - контакты
Command::TYPE_STICKER - стикеры
```

И другие...

### Лучшие практики
Используйте интерфейсы: Всегда реализуйте ContentHandlerInterface для обработчиков

Сохраняйте состояние: Используйте StatefulHandlerInterface для многошаговых сценариев

Управляйте контекстом: Используйте методы работы с контекстом для хранения временных данных

Обрабатывайте ошибки: Всегда обрабатывайте исключения в обработчиках

Используйте цепочки: Для сложных сценариев используйте цепочки обработчиков

Оптимизируйте сессии: Не храните большие объемы данных в сессии

### Отладка
Для отладки используйте логирование:

```php
Yii::info("Обработка команды: /start", 'telegram');
Yii::error("Ошибка обработки: " . $e->getMessage(), 'telegram');
```


