# Установка
### Предпочтительный способ установки этого расширения через composer.

Выполните команду:

```bash
php composer.phar require garmayev/yii2-bot-telegram "*"
```

или добавьте в раздел require вашего composer.json файла:

```json
"garmayev/yii2-bot-telegram": "*"
```

## Настройка
### Базовая конфигурация
Добавьте в конфигурацию приложения (`config/web.php` или `common/config/main.php`):

```php
<?php
return [
    'components' => [
        'telegram' => [
            'class' => 'garmayev\telegram\Telegram',
            'botToken' => 'YOUR_BOT_TOKEN',
        ]
    ]
];
```

### Расширенная конфигурация с командами
```php
<?php
return [
    'components' => [
        'telegram' => [
            'class' => 'garmayev\telegram\Telegram',
            'botToken' => 'YOUR_BOT_TOKEN',
            
            // Регистрация обработчиков команд
            'commands' => [
                '/start' => app\telegram\commands\StartHandler::class,
                '/help' => app\telegram\commands\HelpHandler::class,
                '/menu' => app\telegram\commands\MenuHandler::class,
            ],
            
            // Обработчики контента
            'contentHandlers' => [
                \garmayev\telegram\base\Command::TYPE_TEXT => app\telegram\handlers\TextHandler::class,
                \garmayev\telegram\base\Command::TYPE_PHOTO => app\telegram\handlers\PhotoHandler::class,
                \garmayev\telegram\base\Command::TYPE_DOCUMENT => app\telegram\handlers\DocumentHandler::class,
            ],
            
            // Автоматическое сканирование namespace для поиска обработчиков
            'scanNamespaces' => [
                'app\telegram\commands',
                'app\telegram\handlers',
            ],
            
            // Команды для установки в меню бота
            'botCommands' => [
                ['command' => '/start', 'description' => 'Запустить бота'],
                ['command' => '/help', 'description' => 'Помощь'],
                ['command' => '/menu', 'description' => 'Меню'],
            ],
            
            // Автоматически устанавливать команды при инициализации
            'autoSetBotCommands' => true,
        ]
    ]
];
```

### Список методов API
- text
- getMe
- sendMessage
- forwardMessage
- sendPhoto
- sendAudio
- sendDocument
- sendSticker
- sendVideo
- sendVoice
- sendVideoNote
- sendLocation
- sendChatAction
- getUserProfilePhotos
- getUpdates
- setWebhook
- deleteWebhook
- getChat
- getChatAdministrators
- getChatMembersCount
- getChatMember
- answerCallbackQuery
- editMessageText
- editMessageCaption
- editMessageLiveLocation
- stopMessageLiveLocation
- sendGame
- sendAnimation
- getGameHighScores
- answerInlineQuery
- kickChatMember
- unbanChatMember
- restrictChatMember
- promoteChatMember
- exportChatInviteLink
- deleteMessage
- deleteChatPhoto
- setChatTitle
- setChatDescription
- pinChatMessage
- unpinChatMessage
- leaveChat
- setChatStickerSet
- deleteChatStickerSet
- getFile
- getFileUrl
- sendMediaGroup
- setMyCommands

## Использование
### Базовое использование

```php
<?php 
// Отправка текстового сообщения
Yii::$app->telegram->sendMessage([
    'chat_id' => $chat_id,
    'text' => 'Привет, мир!'
]);

// Отправка сообщения с inline-клавиатурой
Yii::$app->telegram->sendMessage([
    'chat_id' => $chat_id,
    'text' => 'Выберите действие:',
    'reply_markup' => json_encode([
        'inline_keyboard' => [
            [
                ['text' => 'Обновить', 'callback_data' => 'refresh'],
                ['text' => 'Настройки', 'callback_data' => 'settings']
            ]
        ]
    ])
]);

// Отправка фото
Yii::$app->telegram->sendPhoto([
    'chat_id' => $chat_id,
    'photo' => '/path/to/photo.jpg',
    'caption' => 'Описание фото'
]);
?>
```

## Система команд и обработчиков
### Создание обработчика команды
```php
<?php
namespace app\telegram\commands;

use garmayev\telegram\base\Command;
use garmayev\telegram\base\ContentHandlerInterface;
use yii\base\BaseObject;

class StartHandler extends BaseObject implements ContentHandlerInterface
{
    public static function getCommand()
    {
        return '/start';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => 'Добро пожаловать! Я ваш бот-помощник.',
            'reply_markup' => json_encode([
                'keyboard' => [
                    ['/start', '/help'],
                    ['/menu', '/order']
                ],
                'resize_keyboard' => true
            ])
        ]);
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

### Обработчик с состоянием и контекстом
```php
<?php
namespace app\telegram\commands;

use garmayev\telegram\base\Command;
use garmayev\telegram\base\ContentHandlerInterface;
use garmayev\telegram\base\StatefulHandlerInterface;
use yii\base\BaseObject;

class RegistrationHandler extends BaseObject implements ContentHandlerInterface, StatefulHandlerInterface
{
    private $state = [];

    public static function getCommand()
    {
        return '/register';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function setState(array $state)
    {
        $this->state = $state;
    }

    public function getState()
    {
        return $this->state;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        $step = $this->state['step'] ?? 0;

        switch ($step) {
            case 0:
                $this->state['step'] = 1;
                Command::setContext('awaiting_name', true);
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => 'Введите ваше имя:'
                ]);
            
            case 1:
                $name = $args[0] ?? '';
                $this->state['name'] = $name;
                $this->state['step'] = 2;
                Command::removeContext('awaiting_name');
                Command::setContext('awaiting_email', true);
                
                return $telegram->sendMessage([
                    'chat_id' => $telegram->getChatId(),
                    'text' => "Спасибо, {$name}! Теперь введите ваш email:"
                ]);
        }
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

### Обработчик контента (фото)
```php
<?php
namespace app\telegram\handlers;

use garmayev\telegram\base\Command;
use garmayev\telegram\base\ContentHandlerInterface;
use yii\base\BaseObject;

class PhotoHandler extends BaseObject implements ContentHandlerInterface
{
    public static function getCommand()
    {
        return null;
    }

    public static function getContentType()
    {
        return Command::TYPE_PHOTO;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        return null;
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        // $contentData содержит массив с информацией о фото
        $photo = end($contentData); // Берем фото наибольшего размера
        $fileId = $photo->file_id;
        
        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => "Спасибо за фото! File ID: {$fileId}"
        ]);
    }
}
```

### Работа с контекстом
Система предоставляет мощный механизм контекста для хранения данных между запросами:

```php
// Сохранение значения в контекст
Command::setContext('current_step', 'awaiting_phone');

// Получение значения из контекста
$step = Command::getContext('current_step', 'default_value');

// Удаление значения
Command::removeContext('current_step');

// Получение всего контекста
$context = Command::getAllContext();

// Очистка всего контекста
Command::clearContext();
```

### Использование в Controller
Для работы с вебхуком Telegram необходимо отключить CSRF-валидацию в контроллере:

```php
class BotController extends Controller
{
    public $enableCsrfValidation = false;

    public function actionWebhook()
    {
        // Обработка входящих сообщений
        $result = \garmayev\telegram\base\Command::handle();
        
        return $result;
    }
    
    public function actionSetCommands()
    {
        // Установка команд меню бота
        if (Yii::$app->telegram->setBotCommands()) {
            echo "Команды успешно установлены!";
        } else {
            echo "Ошибка установки команд!";
        }
    }
}
```

## Примеры кода
### Получение chat_id

```php
$chatId = Yii::$app->telegram->input->message->chat->id;
```
### Обработка callback запросов

```php
class MenuHandler implements ContentHandlerInterface
{
    public static function getCommand()
    {
        return '/menu';
    }

    public static function getContentType()
    {
        return Command::TYPE_TEXT;
    }

    public function execute($telegram, $args, $isCallback, $callbackQuery)
    {
        if ($isCallback) {
            return $this->handleCallback($telegram, $callbackQuery);
        }

        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => 'Выберите действие:',
            'reply_markup' => json_encode([
                'inline_keyboard' => [
                    [
                        ['text' => 'Опция 1', 'callback_data' => 'option_1'],
                        ['text' => 'Опция 2', 'callback_data' => 'option_2']
                    ]
                ]
            ])
        ]);
    }

    private function handleCallback($telegram, $callbackQuery)
    {
        $data = $callbackQuery->data;
        
        // Ответ на callback запрос
        $telegram->answerCallbackQuery([
            'callback_query_id' => $callbackQuery->id,
            'text' => 'Обработано'
        ]);

        return $telegram->sendMessage([
            'chat_id' => $telegram->getChatId(),
            'text' => "Вы выбрали: {$data}"
        ]);
    }

    public function handle($telegram, $contentData, $isCallback, $callbackQuery)
    {
        return null;
    }
}
```

### Консольные команды
Создайте контроллер для управления командами бота:

```php
namespace app\commands;

use yii\console\Controller;
use yii\console\ExitCode;

class TelegramController extends Controller
{
    /**
     * Установка команд меню бота
     */
    public function actionSetCommands()
    {
        $this->stdout("Установка команд бота...\n");
        
        if (Yii::$app->telegram->setBotCommands()) {
            $commands = Yii::$app->telegram->getBotCommandsConfig();
            $this->stdout("Команды успешно установлены!\n");
            $this->stdout("Установлено команд: " . count($commands) . "\n");
            
            foreach ($commands as $cmd) {
                $this->stdout(" - {$cmd['command']}: {$cmd['description']}\n");
            }
            
            return ExitCode::OK;
        } else {
            $this->stderr("Ошибка установки команд!\n");
            return ExitCode::UNSPECIFIED_ERROR;
        }
    }

    /**
     * Показать зарегистрированные команды
     */
    public function actionListCommands()
    {
        $commands = Yii::$app->telegram->getRegisteredCommands();
        $this->stdout("Зарегистрированные команды: " . count($commands) . "\n");
        
        foreach ($commands as $command) {
            $this->stdout(" - {$command}\n");
        }
        
        return ExitCode::OK;
    }
}
```

### Использование:

```bash
php yii telegram/set-commands
php yii telegram/list-commands
```

## Устаревшая функция Command (для обратной совместимости)

```php
use garmayev\telegram\base\Command;

Command::run("/start", function($telegram){
   $result = $telegram->sendMessage([
      'chat_id' => $telegram->input->message->chat->id,
      "text" => "hello"
   ]);
});
```

### Типы контента

Доступные типы контента для обработчиков:
- `Command::TYPE_TEXT` - текстовые сообщения
- `Command::TYPE_PHOTO` - фото
- `Command::TYPE_VIDEO` - видео
- `Command::TYPE_DOCUMENT` - документы
- `Command::TYPE_AUDIO` - аудио
- `Command::TYPE_VOICE` - голосовые сообщения
- `Command::TYPE_LOCATION` - локации
- `Command::TYPE_CONTACT` - контакты
- `Command::TYPE_STICKER` - стикеры
- `Command::TYPE_ANIMATION` - анимации
- `Command::TYPE_POLL` - опросы
- `Command::TYPE_DICE` - dice

## Лучшие практики

Используйте интерфейсы: Все обработчики должны реализовывать ContentHandlerInterface

Сохраняйте состояние: Для многошаговых сценариев используйте StatefulHandlerInterface

Управляйте контекстом: Используйте методы работы с контекстом для временных данных

Обрабатывайте ошибки: Всегда обрабатывайте исключения в обработчиках

Используйте логирование: Для отладки используйте Yii-логирование

```php
Yii::info("Обработка команды: /start", 'telegram');
Yii::error("Ошибка обработки: " . $e->getMessage(), 'telegram');
```